<?php

declare(strict_types=1);

namespace Gls\GlsPoland\PrestaShop\Grid\Query;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use PrestaShop\PrestaShop\Core\Grid\Query\AbstractDoctrineQueryBuilder;
use PrestaShop\PrestaShop\Core\Grid\Query\DoctrineSearchCriteriaApplicatorInterface;
use PrestaShop\PrestaShop\Core\Grid\Search\SearchCriteriaInterface;

final class ConsignmentQueryBuilder extends AbstractDoctrineQueryBuilder
{
    private $searchCriteriaApplicator;

    public function __construct(Connection $connection, string $dbPrefix, DoctrineSearchCriteriaApplicatorInterface $searchCriteriaApplicator)
    {
        parent::__construct($connection, $dbPrefix);

        $this->searchCriteriaApplicator = $searchCriteriaApplicator;
    }

    public function getSearchQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        $queryBuilder = $this
            ->getCommonQueryBuilder($searchCriteria)
            ->select('c.*, o.reference AS order_reference, GROUP_CONCAT(p.number ORDER BY p.id) AS parcel_numbers')
            ->innerJoin('c', $this->dbPrefix . 'gls_poland_parcel', 'p', 'p.consignment_id = c.id')
            ->groupBy('c.id');

        $this->searchCriteriaApplicator
            ->applyPagination($searchCriteria, $queryBuilder)
            ->applySorting($searchCriteria, $queryBuilder);

        return $queryBuilder;
    }

    public function getCountQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        return $this
            ->getCommonQueryBuilder($searchCriteria)
            ->select('COUNT(*)');
    }

    private function getCommonQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        $queryBuilder = $this->connection
            ->createQueryBuilder()
            ->from($this->dbPrefix . 'gls_poland_consignment', 'c')
            ->innerJoin('c', $this->dbPrefix . 'orders', 'o', 'o.id_order = c.id_order');

        $this->applyFilters($searchCriteria->getFilters(), $queryBuilder);

        return $queryBuilder;
    }

    private function applyFilters(array $filters, QueryBuilder $qb): void
    {
        foreach ($filters as $filterName => $filterValue) {
            switch ($filterName) {
                case 'id':
                case 'id_order':
                    $qb->andWhere('c.`' . $filterName . '` = :' . $filterName);
                    $qb->setParameter($filterName, $filterValue);

                    break;
                case 'sandbox':
                    $qb->andWhere('c.`' . $filterName . '` = :' . $filterName);
                    $qb->setParameter($filterName, $filterValue ? 1 : 0);

                    break;
                case 'tracking_number':
                case 'reference':
                case 'email':
                case 'phone':
                case 'pickup_id':
                    $qb->andWhere('c.`' . $filterName . '` LIKE :' . $filterName);
                    $qb->setParameter($filterName, '%' . $filterValue . '%');

                    break;
                case 'created_at':
                    if (isset($filterValue['from'])) {
                        $qb->andWhere('c.`' . $filterName . '` >= :' . $filterName . '_from');
                        $qb->setParameter($filterName . '_from', sprintf('%s 00:00:00', $filterValue['from']));
                    }

                    if (isset($filterValue['to'])) {
                        $qb->andWhere('c.`' . $filterName . '` <= :' . $filterName . '_to');
                        $qb->setParameter($filterName . '_to', sprintf('%s 23:59:59', $filterValue['to']));
                    }

                    break;
                case 'order_reference':
                    $qb->andWhere('o.`' . $filterName . '` LIKE :' . $filterName);
                    $qb->setParameter($filterName, '%' . $filterValue . '%');

                    break;
                case 'parcel_numbers':
                    $subQuery = $this->connection
                        ->createQueryBuilder()
                        ->select('1')
                        ->from($this->dbPrefix . 'gls_poland_parcel', 'tmp')
                        ->andWhere('tmp.consignment_id = c.id')
                        ->andWhere('tmp.number LIKE :' . $filterName);

                    $qb->andWhere('EXISTS (' . $subQuery . ')');
                    $qb->setParameter($filterName, '%' . $filterValue . '%');

                    break;
                default:
                    break;
            }
        }
    }
}
